/*************************************************************
 *
 *  octree.c - subroutines controlling the octree structure
 *
 *  Copyright (C) 2000,2001  Mark J. Stock, mstock@umich.edu
 *
 *  This file is part of part3d.
 *
 *  part3d is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  part3d is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with part3d; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *********************************************************** */


#include "structs.h"

int add_particle_to_cell(sim_ptr,particle_ptr,cell_ptr);
int split_cell(sim_ptr,cell_ptr);
int count_cells(sim_ptr,cell_ptr);
int replace_all_particles(sim_ptr,cell_ptr,cell_ptr);
int clean_up_all_cells(sim_ptr,cell_ptr);
int clean_up_cells(sim_ptr,cell_ptr,int);
int merge_cell(cell_ptr);
int find_all_cells_cm(sim_ptr,cell_ptr);
int find_cell_cm(cell_ptr,int);


/*
 *  Add the particle to the given cell
 */
int add_particle_to_cell(sim_ptr sim,particle_ptr curr,cell_ptr cell){

   int i,j,k;
   particle_ptr oldfirst;

   if (cell->has_subcells) {
      /* if the cell has subcells, then the particle must go in one of those */
      if (curr->x[0][0] < cell->mid[0]) i = 0;
         else i = 1;
      if (curr->x[0][1] < cell->mid[1]) j = 0;
         else j = 1;
      if (curr->x[0][2] < cell->mid[2]) k = 0;
         else k = 1;
      /* fprintf(stdout,"  putting particle %d in cell %d %d %d\n",curr->index,i,j,k); */
      add_particle_to_cell(sim,curr,cell->s[i][j][k]);
   } else {
      /* otherwise the particle must go in this cell */
      oldfirst = cell->first;
      curr->next = oldfirst;
      cell->first = curr;
      cell->num++;
   }

   if (cell->num > sim->max_parts_in_cell && !cell->has_subcells &&
       cell->level+1 < sim->max_levels) {
      /* now there are too many particles in this cell, split the cell into subcells */
      /* fprintf(stdout,"  cell at level %d has %d particles, splitting\n",cell->level,cell->num); */
      split_cell(sim,cell);
   }

   return(0);
}


/*
 *  Split this cell into 8 subcells, and distribute all particles
 */
int split_cell(sim_ptr sim,cell_ptr parent){

   int i,j,k;
   FLOAT xstart,xend,ystart,yend,zstart,zend;
   FLOAT p[3][3][3];
   particle_ptr curr,thenext;

   /* create the 8 new cells */
   for (i=0;i<2;i++) {
      xstart = parent->min[0] + i*(parent->max[0]-parent->min[0])/2.0;
      xend = parent->min[0] + (i+1)*(parent->max[0]-parent->min[0])/2.0;
      for (j=0;j<2;j++) {
         ystart = parent->min[1] + j*(parent->max[1]-parent->min[1])/2.0;
         yend = parent->min[1] + (j+1)*(parent->max[1]-parent->min[1])/2.0;
         for (k=0;k<2;k++) {
            parent->s[i][j][k] = (CELL*)malloc(sizeof(CELL));
            parent->s[i][j][k]->level = parent->level+1;
            parent->s[i][j][k]->has_subcells = FALSE;
            parent->s[i][j][k]->first = NULL;
            parent->s[i][j][k]->num = 0;
            zstart = parent->min[2] + k*(parent->max[2]-parent->min[2])/2.0;
            zend = parent->min[2] + (k+1)*(parent->max[2]-parent->min[2])/2.0;
            parent->s[i][j][k]->min[0] = xstart;
            parent->s[i][j][k]->min[1] = ystart;
            parent->s[i][j][k]->min[2] = zstart;
            parent->s[i][j][k]->mid[0] = (xstart+xend)/2.0;
            parent->s[i][j][k]->mid[1] = (ystart+yend)/2.0;
            parent->s[i][j][k]->mid[2] = (zstart+zend)/2.0;
            parent->s[i][j][k]->max[0] = xend;
            parent->s[i][j][k]->max[1] = yend;
            parent->s[i][j][k]->max[2] = zend;
            /* fprintf(stdout,"    new cell %d %d %d has coords %g %g  %g %g  %g %g\n",i,j,k,xstart,xend,ystart,yend,zstart,zend); */
         }
      }
   }

   /* then, iterate thru all particles and distribute them */
   /* fprintf(stdout,"    parent cell had %d particles\n",parent->num); */
   curr = parent->first;
   while (curr) {
      thenext = curr->next;
      if (curr->x[0][0] < parent->mid[0]) i = 0;
         else i = 1;
      if (curr->x[0][1] < parent->mid[1]) j = 0;
         else j = 1;
      if (curr->x[0][2] < parent->mid[2]) k = 0;
         else k = 1;
      add_particle_to_cell(sim,curr,parent->s[i][j][k]);
      curr = thenext;
   }
   parent->first = NULL;
   parent->num = 0;

   /*
   for (i=0;i<2;i++)
      for (j=0;j<2;j++)
         for (k=0;k<2;k++) {
            fprintf(stdout,"    child cell %d %d %d now has %d particles\n",i,j,k,parent->s[i][j][k]->num);
            if (parent->s[i][j][k]->first)
               fprintf(stdout,"      one is at %g %g %g\n",parent->s[i][j][k]->first->x[0][0],
                                                           parent->s[i][j][k]->first->x[0][1],
                                                           parent->s[i][j][k]->first->x[0][2]);
         }
   fprintf(stdout,"    parent cell now has %d particles\n",parent->num);
   */

   /* set the flag and we're ready to go */
   parent->has_subcells = TRUE;

   return(0);
}


/*
 *  Fill the cell_count array in sim with an accurate count of the cells
 */
int count_cells(sim_ptr sim,cell_ptr curr_cell) {

   int i,j,k;

   /* count this level */
   sim->cell_count[curr_cell->level]++;

   /* and count all sublevels */
   if (curr_cell->has_subcells)
      for (i=0;i<2;i++)
         for (j=0;j<2;j++)
            for (k=0;k<2;k++)
               count_cells(sim,curr_cell->s[i][j][k]);

   return(0);
}


/*
 *  Replace the particles by completely re-allocating them
 */
int replace_all_particles(sim_ptr sim,cell_ptr top,cell_ptr curr_cell){

   int i,j,k,out_of_bounds;
   particle_ptr curr,last,moving_particle;

   /* for each particle, see if it needs to be moved, if so,
    * move it back up to the top cell where it will reallocate
    * downward to an appropriate cell */

   if (curr_cell->has_subcells) {

      for (i=0;i<2;i++)
         for (j=0;j<2;j++)
            for (k=0;k<2;k++)
               replace_all_particles(sim,top,curr_cell->s[i][j][k]);

   } else {

      curr = curr_cell->first;
      last = curr;
      while (curr) {
         out_of_bounds = FALSE;

         /* check to see if the particle is no longer in the cell's bounds */
         /* if the acc memory is freed, one of these x[0][i]'s can't be seen */
         for (i=0;i<3;i++)
            if (curr->x[0][i] < curr_cell->min[i] || curr->x[0][i] > curr_cell->max[i]) {
               out_of_bounds = TRUE;
               break;
            }

         if (out_of_bounds) {

            /* remove this particle from the current cell */
            /* if (curr->next) fprintf(stdout,"    %d is out of bounds %lf %lf %lf last is %d, next is %d\n",curr->index,curr_cell->min[i],curr->x[0][i],curr_cell->max[i],last->index,curr->next->index); */
            /* else fprintf(stdout,"    %d is out of bounds %lf %lf %lf last is %d, no next\n",curr->index,curr_cell->min[i],curr->x[0][i],curr_cell->max[i],last->index); */
            
            moving_particle = curr;
            if (curr == curr_cell->first) {
               curr_cell->first = curr->next;
               last = curr;
            } else {
               /* fprintf(stdout,"    last is %d, last->next is %d\n",last->index,last->next->index); */
               /* fprintf(stdout,"    curr is %d, curr->next is %d\n",curr->index,curr->next->index); */
               last->next = curr->next;
            }
            curr = last->next;
            curr_cell->num--;

            /* and add it to the top-level cell, but only if it not completely out-of-bounds */
            out_of_bounds = FALSE;
            for (i=0;i<3;i++)
               if (moving_particle->x[0][i] < top->min[i] || moving_particle->x[0][i] > top->max[i]) {
                  out_of_bounds = TRUE;
                  i+=3;
               }
            if (out_of_bounds) {
               // fprintf(stdout,"  particle removed because loc is %g %g %g\n",moving_particle->x[0][0],moving_particle->x[0][1],moving_particle->x[0][2]);
               free(moving_particle);
            } else {
               /* it's OK to add it to the top-level cell */
               add_particle_to_cell(sim,moving_particle,top);
            }


         /* if it's not out of bounds */
         } else {

            /* if (curr->next) fprintf(stdout,"    %d is OK, last is %d, next is %d\n",curr->index,last->index,curr->next->index); */
            /* else fprintf(stdout,"    %d is OK, last is %d, no next\n",curr->index,last->index); */

            /* jump to the next one */
            last = curr;
            curr = curr->next;
         }

      }

   }

   return(0);
}


/*
 *  Then, march backwards from the deepest level and remove
 *  cells with too few particles
 */
int clean_up_all_cells(sim_ptr sim,cell_ptr top){

   int i;

   /* fprintf(stdout,"\n  doing clean_up_all_cells\n"); */

   /* if max_levels is 10, iterate from 9 to 0 */
   for (i=sim->max_levels-1; i>=0; i--) {
      /* fprintf(stdout,"  level %d\n",i); */
      clean_up_cells(sim,top,i);
   }

   /* return 0 if all went well */
   return(0);
}


/*
 *  For this level, either count particles or check to see if the
 *  8 child subcells could be merged together
 */
int clean_up_cells(sim_ptr sim,cell_ptr curr_cell,int do_level){

   int i,j,k,cnt;
   int any_sub_subcells;
   particle_ptr curr;

   /* if there are subcells, then there are no particles attached to this level */
   if (curr_cell->has_subcells) {

      /* if we're to compute this level, then all levels beneath are done */
      if (curr_cell->level == do_level) {

         /* count the particles in the 8 subcells, AND confirm that all
          * subcells have no subcells of their own */
         cnt = 0;
         any_sub_subcells = FALSE;
         for (i=0;i<2;i++)
            for (j=0;j<2;j++)
               for (k=0;k<2;k++) {
                  cnt += curr_cell->s[i][j][k]->num;
                  if (curr_cell->s[i][j][k]->has_subcells) any_sub_subcells = TRUE;
               }

         /* make sure particle count was consistent */
         /* if (cnt != curr_cell->num) {
            fprintf(stderr,"WARNING (clean_up_cells): particle count inconsistent\n");
            fprintf(stderr,"    Count is %d, previous saved count was %d\n",cnt,curr_cell->num);
         } */
         curr_cell->num = cnt;

         /* if there are no sub-subcells AND the particle count is too low, merge */
         if (!any_sub_subcells && cnt<sim->max_parts_in_cell) {
            /* fprintf(stdout,"    merging subcells at level %d, only %d particles\n",curr_cell->level+1,cnt); */
            merge_cell(curr_cell);
         }

      /* otherwise, run this subroutine on all of the subcells */
      } else {
         for (i=0;i<2;i++)
            for (j=0;j<2;j++)
               for (k=0;k<2;k++)
                  clean_up_cells(sim,curr_cell->s[i][j][k],do_level);
      }

   /* and if there are no subcells, doublecheck the particle count */
   } else {

      /* cycle thru all particles, make sure the particle count is accurate */
      curr = curr_cell->first;
      cnt = 0;
      while (curr) {
         cnt++;
         curr = curr->next;
      }

      /* compare this count to the cell's count */
      /* if (cnt != curr_cell->num) {
         fprintf(stderr,"WARNING (clean_up_cells): particle count inconsistent\n");
         fprintf(stderr,"    Count is %d, previous saved count was %d\n",cnt,curr_cell->num);
      } */
      curr_cell->num = cnt;
   }

   return(0);
}


/*
 *  Merge this cell's 8 children back into the cell
 */
int merge_cell(cell_ptr curr_cell){

   int i,j,k;
   particle_ptr scurr,snext,oldfirst;

   /* continuity check on parent cell */
   if (curr_cell->first) {
      fprintf(stderr,"WARNING (merge_cell): parent cell had particles attached to it!\n");
   }

   /* reset this to zero, and add the particles from the subcells */
   curr_cell->num = 0;

   /* then, move all of the 8 subcells' particles to the parent's list */
   for (i=0;i<2;i++)
      for (j=0;j<2;j++)
         for (k=0;k<2;k++) {

            /* loop through the Subcell's CURRent particle */
            scurr = curr_cell->s[i][j][k]->first;
            while (scurr) {

               /* set the Subcell's NEXT particle now */
               snext = scurr->next;

               /* remove the particle from the subcell */
               curr_cell->s[i][j][k]->first = snext;

               /* add the particle to the parent cell */
               oldfirst = curr_cell->first;
               scurr->next = oldfirst;
               curr_cell->first = scurr;
               curr_cell->num++;

               scurr = snext;
            }
         }

   /* set the appropriate pointers */
   curr_cell->has_subcells = FALSE;

   /* free the memory from the 8 subcells */
   for (i=0;i<2;i++)
      for (j=0;j<2;j++)
         for (k=0;k<2;k++) {
            free(curr_cell->s[i][j][k]);
            curr_cell->s[i][j][k] = NULL;
         }

   return(0);
}


/*
 * find the centers of mass of all cells
 */
int find_all_cells_cm(sim_ptr sim,cell_ptr top){

   int i;

   /* if max_levels is 10, iterate from 9 to 0 */
   for (i=sim->max_levels-1; i>=0; i--) {
      /* fprintf(stdout,"\n  doing level %d\n",i); */
      find_cell_cm(top,i);
   }

   /* return 0 if all went well */
   return(0);
}


/*
 * find the centers of mass of all cells
 */
int find_cell_cm(cell_ptr curr_cell,int do_level){

   int do_print = FALSE;
   int i,j,k,l;
   FLOAT com[3];
   FLOAT mtot;
   particle_ptr curr;

   for (i=0;i<3;i++) com[i] = 0.0;
   mtot = 0.0;

   /* if there are subcells, then there are no particles attached to this level */
   if (curr_cell->has_subcells) {

      /* if we're to compute this level, then all levels beneath are done */
      if (curr_cell->level == do_level) {
         curr_cell->num = 0;
         for (i=0;i<2;i++)
            for (j=0;j<2;j++)
               for (k=0;k<2;k++) {
                  mtot += curr_cell->s[i][j][k]->mass;
                  for (l=0;l<3;l++) com[l] += curr_cell->s[i][j][k]->mass * curr_cell->s[i][j][k]->cm[l];
                  curr_cell->num += curr_cell->s[i][j][k]->num;
               }

         /* if, for some reason, total mass is zero, set com to origin */
         if (fabs(mtot) > 1e-6) {
            for (i=0;i<3;i++) com[i] = com[i]/mtot;
            if (do_print) fprintf(stdout,"  level %d cell has mass %g, com %g %g %g\n",curr_cell->level,mtot,com[0],com[1],com[2]);
         } else {
            for (i=0;i<3;i++) com[i] = 0.0;
            if (do_print) fprintf(stdout,"  level %d cell has mass %g\n",curr_cell->level,mtot);
         }

         /* set sums in cell structure */
         curr_cell->mass = mtot;
         for (i=0;i<3;i++) curr_cell->cm[i] = com[i];

      /* otherwise, run this subroutine on all of the subcells */
      } else {
         for (i=0;i<2;i++)
            for (j=0;j<2;j++)
               for (k=0;k<2;k++)
                  find_cell_cm(curr_cell->s[i][j][k],do_level);
      }
   } else {

      /* cycle thru all particles, sum mass and center of mass */
      curr = curr_cell->first;
      while (curr) {
         mtot += curr->mass;
         for (i=0;i<3;i++) com[i] += curr->x[0][i] * curr->mass;
         curr = curr->next;
      }

      /* if, for some reason, the mass is too low, set com to origin */
      if (fabs(mtot) > 1e-6) {
         for (i=0;i<3;i++) com[i] = com[i]/mtot;
         if (do_print) fprintf(stdout,"  level %d particles have mass %g, com %g %g %g\n",curr_cell->level,mtot,com[0],com[1],com[2]);
      } else {
         for (i=0;i<3;i++) com[i] = 0.0;
         if (do_print) fprintf(stdout,"  level %d particles have mass %g\n",curr_cell->level,mtot);
      }

      /* set the sums in the cell structure */
      curr_cell->mass = mtot;
      for (i=0;i<3;i++) curr_cell->cm[i] = com[i];

   }

   return(0);
}

